;;
;; Virtual Disk Drive Project - Nov 12, 2003
;;;			- much of this was taken from sertrack which
;;;			- rewritten extensively in Sep 2003 for DD mostly.
;;

#include "main.h"
#include "vd.h"
#include "utility.h"
#include "serbyte.h"
#include "serbyte.mac"
#include "headmove.mac"
#include "waitP2.mac"
#include "disk.h"
#include "write.h"
#include "write.mac"

;;; =======================================================================
;;; === The following code are the four routines for generating sectors:
;;; === DDBlank, SDBlank, DD, and SD sectors.  Because we ran out of space
;;; === in the first page (when doing DD sectors) all of these are in the
;;; === second page.
;;; =======================================================================

PROG2 CODE
		extern	SerTrackSecDone
		extern	SerTrackFinal

		global	DDWritePrepare
		global	SDWritePrepare
		global	H17WritePrepare
		global	DDBlankSector
		global	SDBlankSector
		global	DDSector
		global	SDSector
		global	H17Sector
		global	DDTrailer
		global	SDTrailer
		
;;;*******************************************************************
;;; NAME:	DDBlankSector() and SDBlankSector()
;;;
;;; DESCR:	Generates a "blank" sector either in DD or SD.  An
;;;		attempt is made to make the sector take up the same
;;;		space as its non-blank counter part.  In this way,
;;;		the disk "rotation" will be the same for each track.
;;;
;;; ARGS:	
;;;
;;; RETURNS:	
;;;
;;; NOTES:	- blank sectors are included in that they take up
;;;		  virtual space on the disk.  They also consume
;;;		  header memory to make things easier (ie - so that
;;;		  a bunch of blank sectors won't cause a huge liability
;;;		  when re-aligning memory)
;;;		- Since this is using the standard "trailer" byte
;;;		  (DD=0x4e/SD=0xff) memory increment is done by simply
;;;		  setting the INCVAR to the right number and then
;;;		  generating the right number of filler bytes.
;;;		  to align memory
;;;		- does a "goto SerTrackSecDone" as its return
;;; 
;;; NOTE - if the gap numbers change, change this too!
;;; 
;;;     DD id gap   = 35 0x4e, 12 0x00, 3 0xa1(spec)  = 50
;;;     DD header   = 0xfe + 4 header + 2 CRC         =  7
;;;	DD data gap = 25 0x4e, 12 0x00, 3 0xa1(spec)  = 40
;;;	DD data     = sector mark + 256 bytes + 2 CRC =  3 + 256
;;;                                                    ----------
;;;	DD TOTAL    = 50 + 7 + 40 + 3 + 256           = 100 + 256
;;; 
;;;     SD id gap   = 14 ff's and 6 zeros             = 20
;;;     SD header   = 0xfe + 4 header + 2 CRC         =  7
;;;	SD data gap = 12 ff's and 6 zeros             = 18
;;;	SD data     = sector mark + 256 bytes + 2 CRC =  3 + 256
;;;                                                    ----------
;;;	SD TOTAL    = 20 + 7 + 18 + 3 + 256           = 58 + 256
;;; 
;;;*******************************************************************
DDBlankSector:
		clrf		ARG0		; (-5)
		call		DD0x4e		; (-4) generate 256 filler bytes

		Lower		INDEX_HOLE	; (-16) down the index hole line
	
		movlw		d'9'		; (-15) 9 memory inc's needed
		movwf		INCVAR		; (-14)
	
		Wait7				; (-13)
	
		movlw		d'100'		; (-6) do more filler bytes
		movwf		ARG0		; (-5)
		call		DD0x4e		; (-4)

		incf		SECCOUNT	; (-3)
	
		PAGESEL		SerTrackSecDone
		goto		SerTrackSecDone	; (-9)
	
SDBlankSector:
		clrf		ARG0		; (-5)
		call		SD0xff		; (-4) generate 256 filler bytes

		Lower		INDEX_HOLE	; (-9) down the index hole line
	
		movlw		d'9'		; (-8) 9 memory inc's needed
		movwf		INCVAR		; (-7)
	
		movlw		d'58'		; (-6) do more filler bytes
		movwf		ARG0		; (-5)
		call		SD0xff		; (-4)

		incf		SECCOUNT	; (-3)
	
		PAGESEL		SerTrackSecDone
		goto		SerTrackSecDone	; (-9)

;;;*******************************************************************
;;; NAME:	SDWritePrepare()
;;;
;;; DESCR:	Called by the SDGap macro, prepares the starting writing
;;;		data because the WRITE_GATE was noticed.  This routine
;;;		simply switches back to the right page for writing, calls
;;;		the write routine, then ends like all good sector routines:
;;;		by jumping back to SerTrackSecDone.
;;;
;;; NOTES:	
;;;*******************************************************************
SDWritePrepare:
		PAGESEL		SDWriteSector
		call		SDWriteSector
		goto		SerTrackSecDone
		
;;;*******************************************************************
;;; NAME:	SDSector()
;;;
;;; DESCR:	Generates an SD sector.
;;;
;;; ARGS:	
;;;
;;; RETURNS:	
;;;
;;; NOTES:	- does a "goto SerTrackSecDone" as its return
;;;		- increases SECCOUNT so the caller doesn't have to
;;;*******************************************************************
#define	SD_PRE_ID	d'14'
#define SD_PRE_DATA	d'12'
		
SDSector:
		movlw	SD_PRE_ID		; (-6)
		movwf	ARG0			; (-5)
		SDGap				; (-4)

		Lower		INDEX_HOLE	; (-9) down the index hole line
	
		;;; do the sector header data

		movlw		0xfe		; (-8) ID address mark
		movwf		ARG1		; (-7)
		movlw		0xc7		; (-6)with special clock
		movwf		ARG2		; (-5)
		call		SByte		; (-4)

		;;; now start pushing out data from memory
		;;; first 6 bytes: trck#, side#, sect#, sect-len, CRC(X2)

		movlw		0xc7		; (-11) reload ARG2 with 0xc7 for later
		movwf		ARG2		; (-10) (it is trashed by Sbyte)

		movlw		0x06		; (-9)
		movwf		STWORK		; (-8) STWORK counts

		;; increase SECCOUNT for the next sector - doing it here
		;; makes using MemAddrInc easy to position to right data block
	
		incf		SECCOUNT	; (-7)
	
SDSector_1:
		Wait2				; (-6)
		MemReadByte			; (-4)
		movwf		ARG1		; (-3)
		call		SByteSimple	; (-2)
		MemLowInc			; (-11)
		decfsz		STWORK		; (-9)
		goto		SDSector_1	; (-8)

	;; allow checks for the WRITE_GATE

		WriteEnable			; (-7)
	
	;; done with the header

		movlw		SD_PRE_DATA	; (-6)
		movwf		ARG0		; (-5)
		SDGap				; (-4) generate ID gap

	;; now the track data itself - which begins by getting the sector mark

		MemReadByte			; (-9) sector mark into W
		movwf		ARG1		; (-8) then into ARG1

;;; It is assumed that ARG2 still has 0xc7 at this point...
;;; no functions above change it
;;; 
;;;  OLD WAY	movlw		0xc7		; () with special clock
;;;		movwf		ARG2		; ()

		MemAddrInc	SECCOUNT	; (-7) position to sector data

		Wait1				; (-5)
		call		SByte		; (-4)
	
		MemLowInc			; (-11) bump mem read pointer

	;; Prepare to push out user data
	;; STWORK is sitting at zero, giving us 256 bytes in the following loop

		Wait3				; (-9)
	
SDSector_2:
		Wait2				; (-6)
		MemReadByte			; (-4)
		movwf		ARG1		; (-3)
		call		SByteSimple	; (-2)
		MemLowInc			; (-11)
		decfsz		STWORK		; (-9)
		goto		SDSector_2	; (-8)

	;; done with user data at this point...now the CRC has to come out
	;; first, we have to flip back to the header sector

		MemAddrLoSet			; (-7)
		Wait1				; (-5)
	
		MemReadByte			; (-4) First CRC Byte
		movwf		ARG1		; (-3)
		call		SByteSimple	; (-2)
	
		MemLowInc			; (-11)

		Wait4				; (-10)
		MemReadByte			; (-6) Second CRC Byte
		movwf		ARG1		; (-5)
		MemLowInc			; (-4)
		call		SByteSimple	; (-2)

		PAGESEL		SerTrackSecDone
		goto		SerTrackSecDone	; (-11) Done with sector

;;;*******************************************************************
;;; NAME:	DDWritePrepare()
;;;
;;; DESCR:	Called by the DDGap macro, prepares the starting writing
;;;		data because the WRITE_GATE was noticed.  This routine
;;;		simply switches back to the right page for writing, calls
;;;		the write routine, then ends like all good sector routines:
;;;		by jumping back to SerTrackSecDone.
;;;
;;; NOTES:	
;;;*******************************************************************
DDWritePrepare:
		PAGESEL		DDWriteSector
		call		DDWriteSector
		goto		SerTrackSecDone
	
;;;*******************************************************************
;;; NAME:	DDSector()
;;;
;;; DESCR:	Generates a DD sector.
;;;
;;; ARGS:	- uses STWORK, STWORK2
;;;
;;; RETURNS:	
;;;
;;; NOTES:	- does a "goto SerTrackSecDone" as its return
;;;		- increases SECCOUNT so the caller doesn't have to
;;;*******************************************************************
#define DD_PRE_ID	d'23'
#define DD_PRE_DATA	d'25'

DDSector:
		movlw	DD_PRE_ID		; (-6)
		movwf	ARG0			; (-5) number of 0x42's to do
		DDGap				; (-4)

		Lower		INDEX_HOLE	; (-13) drop the index hole line

		movlw		0x06		; (-12) 6 header bytes
		movwf		STWORK		; (-11) STWORK counts

		;; do the sector id mark of 0xfe

		movlw		0xfe		; (-10)
		movwf		ARG1		; (-9)
		DDByte		0		; (-8) no memory increment

		;; prepare sector count for next sector, doing it here
		;; makes it easy for MemAddrInc to position to the right block
	
		incf		SECCOUNT	; (-13) position for next sector

			;;; do the sector header data
		;;; now start pushing out data from memory
		;;; first 6 bytes: trck#, side#, sect#, sect-len, CRC(X2)

		Wait2				; (-12)
	
DDSector_1:
		MemReadByte			; (-10)
		movwf		ARG1		; (-9)
		DDByte		1		; (-8) with memory increment
		decfsz		STWORK		; (-13)
		goto		DDSector_1	; (-12)

		MemReadByte			; (-11) get the sector mark for later
		movwf		STWORK2		; (-10)

		MemAddrInc	SECCOUNT	; (-9) position to sector data
	
	;; allow checks for the WRITE_GATE

		WriteEnable			; (-7)

	;; put out the data gap

		movlw	DD_PRE_DATA		; (-6)
		movwf	ARG0			; (-5) number of 0x4e's to generate
		DDGap				; (-4)

	;; NOTE that this routine MAY NOT return from this macro since a write op
	;; will cause it to move completely around the rest of this routine.
	;; NOTE too that the write routine needs to reset MemAddr back to the
	;; base in order to write in the sector mark since MemAddrInc has already been done

		Wait3				; (-13) MemAddrInc ISN'T here because it trashes carry

	;; now the track data itself - which begins by getting the sector mark

		movf		STWORK2,W	; (-10)
		movwf		ARG1		; (-9)
		DDByte		1		; (-8) includes memory increment

	;; STWORK is sitting at zero, so just decr that thing

		Wait2				; (-13)
		decf		STWORK		; (-11) ready for 255 bytes
	
DDSector_2:	;; loop through the 255 bytes
	
		MemReadByte			; (-10)
		movwf		ARG1		; (-9)
		DDByte		1		; (-8) with memory increment
		decfsz		STWORK		; (-13)
		goto		DDSector_2	; (-12)

		Wait1				; (-11)
	
	;; do the last byte separately to save a necessary cycle

		MemReadByte			; (-10)
		movwf		ARG1		; (-9)
		DDByte		1		; (-8) with memory increment

	;; done with user data at this point...now the CRC has to come out
	;; first, we have to flip back to the header sector

		MemAddrLoSet			; (-13)

		Wait1				; (-11)

		MemReadByte			; (-10) CRC byte
		movwf		ARG1		; (-9)
		DDByte		1		; (-8) with memory increment

		Wait3				; (-13)
		MemReadByte			; (-10) CRC byte
		movwf		ARG1		; (-9)
		DDByte		1		; (-8) with memory increment

		PAGESEL		SerTrackSecDone
		goto		SerTrackSecDone	; (-13) Done with sector
	
	
;;;*******************************************************************
;;; NAME:	DDTrailer() & SDTrailer()
;;;
;;; DESCR:	Generates an appropriate DD/SD trailer for the end of the
;;;		track.  Also aligns memory.
;;;
;;; ARGS:	
;;;
;;; RETURNS:	
;;;
;;; NOTES:	- does a "goto SerTrackFinal" as the return.
;;;		- DD/SD trailer is just a series of 0x4e/0xff's
;;;		- assumes that INCVAR is already set to the right value
;;;		  to align memory
;;;*******************************************************************
DDTrailer:
		MemReadByte		; (-8) get the number of trailer inc's needed
		movwf	INCVAR		; (-7) place it in right place
					; Note that INCVAR needs to increment
					;	for this byte too
		movlw	d'40'		; (-6) randomly choose 40 bytes "filler"
		movwf	ARG0		; (-5)
		call	DD0x4e		; (-4)
		PAGESEL	SerTrackFinal
		goto	SerTrackFinal	; (-15)
	
SDTrailer:
		MemReadByte		; (-8) get the number of trailer inc's needed
		movwf	INCVAR		; (-7) place it in right place
					; Note that INCVAR needs to increment
					;	for this byte too
		movlw	d'20'		; (-6) 75 seemed to work last time
		movwf	ARG0		; (-5)
		call	SD0xff		; (-4)
		PAGESEL	SerTrackFinal
		goto	SerTrackFinal	; (-9)

;;;*******************************************************************
;;; NAME:	H17WritePrepare()
;;;
;;; DESCR:	Called by the 0x00 routine, prepares the starting writing
;;;		data because the WRITE_GATE was noticed.  This routine
;;;		simply switches back to the right page for writing, calls
;;;		the write routine, then ends like all good sector routines:
;;;		by jumping back to SerTrackSecDone.
;;;
;;; NOTES:	
;;;*******************************************************************
H17WritePrepare:
		PAGESEL		H17WriteSector
		call		H17WriteSector
		goto		SerTrackSecDone
	
;;;*******************************************************************
;;; NAME:	H17Sector()
;;;
;;; DESCR:	Generates a sector emulating a Heathkit H17.  These
;;;		are hard sectored puppies.  Note that there is no
;;;		trailer for these floppies.
;;;
;;; ARGS:	
;;;
;;; RETURNS:	
;;;
;;; NOTES:	The byte layout for an H17 sector header is:
;;;			(1) Volume
;;;			(2) Track
;;;			(3) Sector
;;;			(4) Header CRC byte
;;;			(5) Data CRC byte
;;;			(6-9) unused
;;;		The sector is generated as follows:
;;;			- Index pulse (hard sectoring)
;;;			- XX bytes of zero
;;;			- lower Index (needs to be XXX ms)
;;;			- XX more bytes of zero
;;;			- sync byte
;;;			- volume
;;;			- track
;;;			- sector
;;;			- header CRC
;;;			- 12 bytes of zero
;;;			- snyc byte
;;;			- 256 bytes of data
;;;			- data CRC
;;;		Index pulses are 4 ms long (+- 1.5ms)
;;;			- rotational speed is 300 RPM
;;;			- 10 sectors evenly spaced around the disk
;;;			- 300 R/M * 10 S/R = 300*10 RS/MR = 3000 S/M
;;;			- 3000 S/M / 60 s/M = 3000/60 S/s = 50 S/s = 1/50 s/S = .020 s/S
;;;		Space between sector index pulses is 20 ms
;;;			- 20 ms / sector = 200 ms/R
;;;			- each sector needs to be 20 ms in total
;;;			- includes 2 sync bytes, 3 address, and 2 CRC = 7
;;;			- 256 data bytes = 256
;;;			- bytes are 62.5 us long, or 7.8125 us per bit
;;;			- 7 + 12 + 256 = 275 internal bytes
;;;			- need : 20 ms / 62.5 us/byte = 320 bytes
;;;			- 320 - 275 = 45 .... need 45 "extra" bytes
;;;		Index holes need to be:
;;;			- 4 ms / 62.5 us/byte = 64 bytes'ish
;;;			- we use the +- to drop this down to 45 bytes...
;;;		Space between track index pulse and track 0 is 10 ms
;;;		Since the first sector (0) is called with the index
;;;		pulse raised by the sertack() routine, we need to wait
;;;		for 10 ms before jumping into the sector.
;;;			- 10 ms / 62.5 us/byte = (/ 10 .0625) = 160 bytes
;;;		this is going to make the tracks longer than they
;;;		should be because normally the track index would occur
;;;		during the generation of track 9...which is really what
;;;		needs to occur...we may need to move the raising of the
;;;		track index to inside of these routines...
;;;     S       S   T   S       S       S
;;;     8       9   P   0       1       2
;;;		It should be such that the TP comes half-way through
;;;		track 9...or around 100 bytes or so into it.
;;;		Why don't we just ensure that the INDEX is raised when
;;;		entering the sector routine (even though it may have
;;;		been already raised for sector 0) and then raise it
;;;		again in the middle of sector 9?  Maybe we slide the
;;;		tracks logically sideways so that the first track that
;;;		comes out is 9?  Hummmm....
;;;*******************************************************************
#define H17_PRE_ID	d'13'
#define H17_PRE_DATA	d'18'

;; the h17 sync byte is 0xfd, but it is sent out LSb first
;; 
;; #define REAL_H17_SYNC_BYTE	0xfd

#define H17_SYNC_BYTE	0xbf

SDDataLoop:	macro	var
		Wait2				; (-6) target of goto below
		MemReadByte			; (-4)
		movwf		ARG1		; (-3)
		call		SByteSimple	; (-2)
		MemLowInc			; (-11)
		decfsz		var		; (-9)
		goto		$-d'8'		; (-8) target above
		endm				; (-7)
	
H17Sector:
	;; if SECCOUNT is 9, then (since we're in SD - 10 sector mode)
	;; then this is the last sector and the index mark should be
	;; generated.  STWORK2 tracks this case.

		movlw		d'9'		; (-15)
		subwf		SECCOUNT,W	; (-14)
		movwf		STWORK2		; (-13) zero if last sector

	;; prepare to increment memory during zero gap...this is done in h17
	;; here, as opposed to in the trailer with most other formats
	;; we do 19 increments for sects 1-9 and 24 for sector 10

		movlw		d'19'		; (-12)
		movf		STWORK2		; (-11)
		skpnz				; (-10)
		addlw		d'5'		; (-9)
		movwf		INCVAR		; (-8)
	
	;; Hard sectoring means that every sector raises this
	;; though it may already be up

		Raise		INDEX_HOLE	; (-7)

	;; go ahead and increment SECCOUNT here...it needs to be incremented
	;; for the next sector, but by doing it here it makes it easy to
	;; move to the right data block with MemAddrInc

		incf		SECCOUNT	; (-6)
	
	;; now do the pre-header zero space

		Wait1				; (-5)
		movlw	H17_PRE_ID		; (-4)
		movwf	ARG0			; (-3)
		call	SD0x00			; (-2)
	
	;;; do the sector header data

		Wait3				; (-9)
		movlw		H17_SYNC_BYTE	; (-6)
		movwf		ARG1		; (-5)
		call		SByteSimple	; (-4) send out sync byte

		movlw		d'4'		; (-11) vol#, trck#, sec#, CRC byte
		movwf		STWORK		; (-10)
		Wait3				; (-9)
		SDDataLoop	STWORK		; (-6)

	;; allow checks for the WRITE_GATE

		WriteEnable			; (-7)

	;; ARG1 is loaded with the sync byte for use below - SD0x00 doesn't touch it
	
		movlw		H17_SYNC_BYTE	; (-6) 
		movwf		ARG1		; (-5)

	;; now send out pre data zeros
	
		movlw		H17_PRE_DATA	; (-4)
		movwf		ARG0		; (-3)
		call		SD0x00		; (-2)
		
	;; a WRITE_GATE high will cause early termination of the SD0x00 generation
	;; check for it here
	
		SkipIfLow WRITE_GATE		;(-9)
		SkipIfWriteEnabled		;(-8)
		SkipAlways			;(-7)
		goto	H17WritePrepare		;(-6)    then jump to writing code

	;; otherwise, go on with life

		Wait1				; (-5) ARG1 is already loaded above
		call		SByteSimple	; (-4)

		MemAddrInc	SECCOUNT	; (-10) position to sector data

	;; the strategy here is to do:
	;;	32 bytes
	;;	lower the track index pulse
	;;      96 bytes
	;;	raise the track index pulse if necessary
	;;	64 bytes
	;;	lower the track index pulse
	;;	64 last bytes

		movlw		d'31'		; (-8)
		movwf		STWORK		; (-7) prepare for 32 initial bytes
						;      the last one comes out separately
		SDDataLoop	STWORK		; (-6)

		Wait3				; (-7)

	;; do the 32'nd byte outside the loop to gain a couple cycles
	
		MemReadByte			; (-4) 
		movwf		ARG1		; (-3)
		call		SByteSimple	; (-2)
		MemLowInc			; (-11)
	
	;; lower the track index pulse

		Lower		INDEX_HOLE	; (-9)
	
		movlw		d'95'		; (-8)
		movwf		STWORK		; (-7)
		SDDataLoop	STWORK		; (-6)

		movlw		d'63'		; (-7) prepare for 64 bytes
		movwf		STWORK		; (-6)
		Wait1				; (-5)
	
	;; do the 96'th byte outside the loop to gain a couple cycles
	
		MemReadByte			; (-4) 
		movwf		ARG1		; (-3)
		call		SByteSimple	; (-2)
		MemLowInc			; (-11)
	
	;; bring up the track index pulse here if needed

		movf		STWORK2		; (-9)
		skpnz				; (-8)
		Raise		INDEX_HOLE	; (-7)

		SDDataLoop	STWORK		; (-6)

		Wait3				; (-7)
	
	;; do the 64'th byte outside the loop to gain a couple cycles
	
		MemReadByte			; (-4) 
		movwf		ARG1		; (-3)
		call		SByteSimple	; (-2)
		MemLowInc			; (-11)

		Lower		INDEX_HOLE	; (-10)
	
		movlw		d'63'		; (-9)
		movwf		STWORK		; (-8) prepare for 64 more bytes
						;      the last one comes out separately
		Wait1				; (-7)

		SDDataLoop	STWORK		; (-6)

		Wait3				; (-7)
	
	;; do the 64'th byte outside the loop to gain a couple cycles
	
		MemReadByte			; (-4) 
		movwf		ARG1		; (-3)
		call		SByteSimple	; (-2)
		MemLowInc			; (-11)
	
	;; done with user data at this point...now the CRC has to come out
	;; but first, we have to flip back to the header sector

		MemAddrLoSet			; (-9) flip back to header
		Wait1				; (-7)
	
		MemReadByte			; (-6)
		movwf		ARG1		; (-5)
		call		SByteSimple	; (-4) send out Data CRC Byte
		MemLowInc			; (-11)

		Wait4				; (-9)

	;; the "Tail" gets memory aligned right for the next sector
	;; if writing, this is where we jump back into mainline code
	
H17SectorTail:

		WriteDisable			; (-5)
	
	;; since the header is shorter than originally planned, put a trailer
	;; of 57 - PRE_ID - PRE_DATA

		movlw		d'57'-H17_PRE_DATA-H17_PRE_ID	; (-4)
		movwf		ARG0		; (-3)
		call		SD0x00		; (-2)
	
	;; Note that memory for the h17 format is aligned NOT with a trailer
	;; but during the generation of the track itself!
	
		PAGESEL		SerTrackSecDone	; (-9)
		goto		SerTrackSecDone	; (-11) Done with sector

;;; =======================================================================
;;; END of the PROG2 section
;;; =======================================================================
	END
